*-------------------------------------------------------------------------------
/* title : Summary statistics of firm-level data
   source: ORBIS-Amadeus (proprietary firm-level data)	 
   input : $temp/Sample.dta and $temp/Selection_criteria.dta
   output: $output_tex/OnlineAppendix_TableD1
		   $output_tex/Table1a
		   $output_tex/OnlineAppendix_TableD2a 
		   $output_tex/OnlineAppendix_TableD2b
		   $output_fig/OnlineAppendix_FigureD3
		   $output_fig/OnlineAppendix_FigureD4 */
*-------------------------------------------------------------------------------

*===============================================================================
* Set up
*===============================================================================
*--------------------------SET-UP & READ DIRECTORIES----------------------------
cd												// Current directory of code
do "[0.0] directories.do"

*--------------------------------FIGURE STYLE-----------------------------------
* Graph scheme
set scheme s2color // s2color cleanplots 
grstyle init
grstyle set plain, nogrid noextend horizontal // compact
grstyle set legend 6
grstyle set legend, nobox
graph set window fontface "Times New Roman"

* File-types to export graphs
global filetype	pdf // pdf png eps

*===============================================================================
* Online Appendix Table D1 - Summary statistics of firm-level selection criteria
*===============================================================================
use "$temp/Selection_criteria.dta", clear

foreach criterion in consolidation legalstatus missing L20 bacon30 drop3bacon30 bacon15 drop3bacon15 {
	foreach stat in N F Y L {
		if "`criterion'"=="bacon15" gen `stat'_`criterion'=(`stat'1_`criterion'/`stat'0_bacon30)*100
		else 						gen `stat'_`criterion'=(`stat'1_`criterion'/`stat'0_`criterion')*100
	}
}

drop *0_* *1_*
greshape long N_ F_ Y_ L_, i(country) j(criterion) s
rename *_* **

gen sort = 0
local i = 1
foreach criterion in legalstatus consolidation missing L20 bacon30 drop3bacon30 bacon15 drop3bacon15 {
	replace sort = `i' if criterion=="`criterion'"
	local ++i
}

sort sort
rename N O
estpost tabstat O F Y L, by(sort) nototal 
estout . using "$output_tex/OnlineAppendix_TableD1.tex", replace style(tex) mlabels(, none) collabels(, none) eqlabels(, none) 	///
				cells("O(fmt(%9.2f)) F(fmt(%9.2fc)) Y(fmt(%9.2fc)) L(fmt(%9.2fc))") 											///
				rename(1 "1. Active Legal Status" 2 "2. Consolidated Accounts" 3 "3. No Missing Data" 4 "4. $>$20 Employees" 	/// 
					   5 "5.i. BACON 30th percentile" 6 "5.ii. $>$2 consecutive observations" 7 "6.i. BACON 15th percentile" 	///
					   8 "6.ii. $>$2 consecutive observations")

*===============================================================================
* Table 1a - Summary statistics of firm-level variables
*===============================================================================
use id year CPA SALES TNGFA MCOST L shh_dom shh_for sub_dom sub_for using "$temp/Sample.dta", clear
gen all = 1

foreach var in SALES TNGFA MCOST {
	replace `var' = `var'/1000 				// in thousand Euro
}

estpost tabstat SALES TNGFA MCOST L sub_dom sub_for shh_dom shh_for, by(all) stats(count mean sd min p25 p50 p75 max) nototal columns(statistics)
estout . using "$output_tex/Table1a.tex", replace style(tex) mlabels(, none) collabels(, none) eqlabels(, none) 									///
				rename(SALES "$\!\text{Sales}^{\dagger}\$" TNGFA "$\!\text{Tang. Fixed Assets}^{\dagger}\$" MCOST "$\!\text{Material}^{\dagger}\$" 	///
					  L "$\!\text{Number of Employees}\$" sub_dom "$\!{SUB}^{dom}\$" sub_for "$\!{SUB}^{for}\$" shh_dom "$\!{SHH}^{dom}\$" 			///
					  shh_for "$\!{SHH}^{for}\$") substitute("\!" "") 																				///
				cells("count(fmt(%9.0fc)) mean(fmt(%9.1gc)) sd(fmt(%9.1gc)) min(fmt(%9.1gc)) p25(fmt(%9.1gc)) p50(fmt(%9.1gc)) p75(fmt(%9.1gc)) max(fmt(%9.1gc))")

*===============================================================================
* Online Appendix Table D2a & D2b - Summary statistics of pf estimates
*===============================================================================
local pf a_k a_l a_m RTS

* Imperfect Competition
use id year CPA `pf' using "$output_dta/TFP_IC.dta", clear
foreach var of local pf {
	rename `var' `var'_IC
}

* Baseline
merge 1:m id year using "$output_dta/TFP_Baseline.dta", nogen keepusing(`pf')

* Number of observations by industry
gegen Obs = count(a_k), by(CPA)
gen all = 1

* Table D2a (per industry)
estpost tabstat Obs a_k a_l a_m RTS a_k_IC a_l_IC a_m_IC RTS_IC, by(CPA) nototal stats(mean) 
estout . using "$output_tex/OnlineAppendix_TableD2a.tex", replace style(tex) mlabels(, none) collabels(, none) eqlabels(, none) varlabels(Total "Mean") ///
				cells("Obs(fmt(%9.0fc)) a_k(fmt(%9.3fc)) a_l(fmt(%9.3fc)) a_m(fmt(%9.3fc)) RTS(fmt(%9.3fc)) a_k_IC(fmt(%9.3fc)) a_l_IC(fmt(%9.3fc)) a_m_IC(fmt(%9.3fc)) RTS_IC(fmt(%9.3fc))")

* Table D2a (total)
drop Obs
gegen Obs = count(a_k)
estpost tabstat Obs a_k a_l a_m RTS a_k_IC a_l_IC a_m_IC RTS_IC, by(all) nototal stats(mean median sd) 
estout . using "$output_tex/OnlineAppendix_TableD2b.tex", replace style(tex) mlabels(, none) collabels(, none) eqlabels(, none) varlabels(mean "All-mean" p50 "All-median" sd "All-st.dev.") ///
				cells("Obs(fmt(%9.0fc)) a_k(fmt(%9.3fc)) a_l(fmt(%9.3fc)) a_m(fmt(%9.3fc)) RTS(fmt(%9.3fc)) a_k_IC(fmt(%9.3fc)) a_l_IC(fmt(%9.3fc)) a_m_IC(fmt(%9.3fc)) RTS_IC(fmt(%9.3fc))")

*===============================================================================
* Online Appendix Figure D3 - Markup by year
*===============================================================================
use year markup using "$output_dta/TFP_IC.dta" , replace
sort year

* Plot figure
twoway line markup year, xtitle("") ylabel(1.04(.01)1.10, format(%-9.2f)) xlab(2001 2004 2008 2011 $yearf) ///
			lwidth(thin) lcolor(black) graphregion(margin(-13.5 2 -4 -4)) name(g1,replace) 

* Export figure
foreach gtype of global filetype {
	graph export "$output_fig/OnlineAppendix_FigureD3.`gtype'", replace
}

graph drop _all

*===============================================================================
* Online Appendix Figure D4 - Kernel densities of bootstrapped standard errors
*===============================================================================
use downIM upEX upIM downEX using "$output_dta/Baseline.dta", clear

* Rename variables
rename *IM *IM_jc
rename *EX *EX_jc

foreach cluster in j c i {
	merge 1:1 _n using "$output_dta/Cluster_`cluster'.dta", keepusing(downIM upEX upIM downEX) nogen
	rename *IM *IM_`cluster'
	rename *EX *EX_`cluster'
}

local i=1
 foreach var in downIM upEX upIM downEX {

	preserve
		use "$output_dta/TFP_Baseline.dta", clear
		rename *0d* **
		qui sum rho_`var', mean
	restore
	
	twoway kdensity `var'_jc,lpattern(solid) 			lwidth(thin) lcolor(black) 		///
		|| kdensity `var'_j ,lpattern(longdash) 		lwidth(thin) lcolor(black) 		///
		|| kdensity `var'_c ,lpattern(shortdash) 		lwidth(thin) lcolor(black) 		///
		|| kdensity `var'_i ,lpattern(shortdash_dot) 	lwidth(thin) lcolor(black) ||, 	///
		xtitle("") ylab(minmax, format(%-9.0f)) xlab(-1(0.5)1) xline(`r(mean)') 		///
		legend(lab(1 "jc") lab(2 "j") lab(3 "c") lab(4 "i") rows(1) symxsize(10)) 		/// 
		title("`var'", color(black)) graphregion(margin(zero)) name(g`i',replace)
		
	local ++i
}

* Combine sub-figures
grc1leg g1 g2 g3 g4, legendfrom(g1) graphregion(margin(zero)) imargin(-8 0 0 0)

* Export figure
foreach gtype of global filetype {
	graph export "$output_fig/OnlineAppendix_FigureD4.`gtype'", replace
}

graph drop _all
